# Spread the data -------------------------------------------------------------------------------------------------
# Spread the data from long to wide format, where columns are time, and rows are Units

spread.data <- function(Data, select_columns = c("UnitID", "norm.time", "z.score")) {
  tryCatch({
    
    # spread the data for wide format
    Data.spread <- Data %>% 
      select_at(select_columns) %>%
      spread(key = "norm.time", value = "z.score")
    
    return(Data.spread)
    
  }, error = function(e) {
    warning("ERROR!")
    warning(e)
    print(stat.data)
  })
}

# Permute data ----------------------------------------------------------------------------------------------------
# Permute the data by multiplying the entire Unit time series by 1 or -1

permute.data  <- function(Data) {
  tryCatch({
    
    vect <- sample(c(-1, 1), nrow(Data), replace = TRUE)
    
    Data <- Data %>% mutate_if(is.numeric, ~ .x * vect)
    
    return(Data)
    
  }, error = function(e) {
    warning("ERROR!")
    warning(e)
    print(stat.data)
  })
}


# One sample t-test -----------------------------------------------------------------------------------------------
# Perform one sample t.test for entire time series, comparing to 0, works on the long time series data, where
# columns should be time, and rows should be Units

p.val.one.group <- function(Data) {
  tryCatch({
    
    stat.data  <- data.frame(T.val = NA,
                             p.val = NA,
                             mean = NA,
                             time = NA,
                             cluster_sign = NA,
                             cluster_nr = NA,
                             sum.t = NA,
                             n = NA)
    
    if (nrow(Data) > 2 & all(!is.na(Data %>% select_if(is.numeric)))) {
        # calculate p value one-sample
        
        p.val <- map(Data %>% select_if(is.numeric), t.test, mu = 0)
        
        stat.data <- data.frame(
          T.val = sapply(p.val, function(x) as.numeric(x$statistic)),
          p.val = sapply(p.val, function(x) as.numeric(x$p.value)),
          mean = sapply(p.val, function(x) as.numeric(x$estimate)),
          time = names(p.val)) 
        
        t.threshold <- qt(c(.025, .975), df = nrow(Data) - 1)
        
        stat.data <- stat.data %>% 
          mutate(positive = if_else(T.val > max(t.threshold), 1, 0),
                  negative = if_else(T.val < min(t.threshold), 1, 0),
                  positive_diff = if_else(c(0, diff(positive)) > 0, 1, 0),
                  negative_diff = if_else(c(0, diff(negative)) > 0, 1, 0),
                  cluster_nr = cumsum(positive_diff + negative_diff),
                  cluster_sign = case_when(positive == 1 ~ "positive",
                                            negative == 1 ~ "negative"),
                 sum.t = NA, n = NA) %>%
          filter(cluster_sign %in% c("positive", "negative"))

        if (is_empty(stat.data)) {
          return(stat.data %>% 
                   select(T.val, p.val, mean, time, cluster_sign, cluster_nr, sum.t, n))
          print("No significant clusters")
          
        } else {
          
          stat.data <- stat.data %>% 
            group_by(cluster_nr) %>%
            mutate(sum.t = sum(T.val), n = n()) %>% 
            ungroup() %>% 
            select(T.val, p.val, mean, time, cluster_sign, cluster_nr, sum.t, n)
          
          return(stat.data)
        }
        
      } else {
        #print("All NA/Empty")
        return(stat.data)
        
      }
      

  }, error = function(e) {
    warning("ERROR!")
    warning(e)
    print(stat.data)
  })
}


# One.sample.iteratation --------------------------------------------------
# Main function to run one sample t.test, taking the above functions

One.sample.iteration <- function(iter, Data,
                                 group_columns = c("Area.brain", "TreatmentID", "Event"),
                                 select_columns = c("UnitID", "norm.time", "z.score"), 
                                 permutation = FALSE) {
  
  print(iter)
  
  Data.iter <- Data %>%
    group_by_at(group_columns) %>%
    nest() %>%
    mutate(spread = map(data, spread.data, select_columns = select_columns))
  
  if (permutation) {
    
    Data.iter <- Data.iter %>%
      mutate(spread = map(spread, permute.data))
    
  }
    
  Data.iter <- Data.iter %>%
    mutate(p.val = map(spread, p.val.one.group)) %>%
    select(-data, -spread) %>%
    unnest() %>% ungroup() %>% 
    mutate(iteration = iter)
  
  return(Data.iter)
}

# Two sample t-test -----------------------------------------------------------------------------------------------
# Perform two sample t.test for entire time series, comparing to 0, works on the long time series data, where
# columns should be time, and rows should be Units, needs Event column to use as grouping

p.val.two.group <- function(Data, paired.test = TRUE) {
  tryCatch({
    
    stat.data  <- data.frame(T.val = NA,
                             p.val = NA,
                             time = NA,
                             cluster_sign = NA,
                             cluster_nr = NA,
                             sum.t = NA,
                             n = NA)
    
    if (nrow(Data) > 2 & all(!is.na(Data %>% select_if(is.numeric))) & length(unique(Data$Event)) == 2) {
      # calculate p value one-sample
      
      columns <- Data %>% select_if(is.numeric) %>% colnames(.)
      
      p.val <- map(columns, ~t.test(as.formula(paste0("`", .x, "`", "~ Event")), 
                                                          data = Data, 
                                    paired = paired.test, 
                                    var.equal = TRUE))
      
      stat.data <- data.frame(
        T.val = sapply(p.val, function(x) as.numeric(x$statistic)),
        p.val = sapply(p.val, function(x) as.numeric(x$p.value)),
        time = map_chr(p.val, function(x) x$data.name)) # mean_diff = sapply(p.val, function(x) diff(as.numeric(x$estimate)))
      
      t.threshold <- qt(c(.025, .975), df = nrow(Data) - 2)
      
      stat.data <- stat.data %>%
        mutate(time = str_replace(time, " by Event", "")) %>%
        mutate(positive = if_else(T.val > max(t.threshold), 1, 0),
               negative = if_else(T.val < min(t.threshold), 1, 0),
               positive_diff = if_else(c(0, diff(positive)) > 0, 1, 0),
               negative_diff = if_else(c(0, diff(negative)) > 0, 1, 0),
               cluster_nr = cumsum(positive_diff + negative_diff),
               cluster_sign = case_when(positive == 1 ~ "positive",
                                        negative == 1 ~ "negative"),
               sum.t = NA, n = NA) %>%
        filter(cluster_sign %in% c("positive", "negative"))
      
      if (is_empty(stat.data)) {
        return(stat.data %>% 
                 select(T.val, p.val, time, cluster_sign, cluster_nr, sum.t, n))
        print("No significant clusters")
        
      } else {
        
        stat.data <- stat.data %>% 
          group_by(cluster_nr) %>%
          mutate(sum.t = sum(T.val), n = n()) %>% 
          ungroup() %>% 
          select(T.val, p.val, time, cluster_sign, cluster_nr, sum.t, n)
        
        return(stat.data)
      }
      
    } else {
      #print("All NA/Empty")
      return(stat.data)
      
    }
    
    
  }, error = function(e) {
    warning("ERROR!")
    warning(e)
    print(stat.data)
  })
}


# Permute labels --------------------------------------------------------------------------------------------------

permute.labels <- function(Data, grouping_column, permuted_column){
  
  if(!is_empty(grouping_column)){
    Data <- Data %>% group_by_at(grouping_column)
  }
  
  Data <- Data %>%
    mutate_at(permuted_column, sample) %>%
    ungroup()
  
  return(Data)

}

# Two.sample.iteratation --------------------------------------------------
# Main function to run one sample t.test, taking the above functions

two.sample.iteration <- function(iter, Data,
                                 filter_events = c("Freezing_80_start", "Precue_Pokes"),
                                 group_columns = c("Area.brain", "TreatmentID"),
                                 select_columns = c("Event", "UnitID", "norm.time", "z.score"), 
                                 grouping_column_permutation = "UnitID",
                                 permuted_column_permutation = "Event",
                                 permutation = FALSE,
                                 paired.test = FALSE) {
  
  print(iter)
  
  Data.iter <- Data %>%
    filter(Event %in% filter_events) %>%
    group_by_at(group_columns) %>%
    nest() %>%
    mutate(spread = map(data, spread.data, select_columns = select_columns))
  
  if (permutation) {
    
    Data.iter <- Data.iter %>%
      mutate(spread = map(spread, permute.labels, 
                          grouping_column = grouping_column_permutation,
                          permuted_column = permuted_column_permutation))
    
  }
  
  Data.iter <- Data.iter %>%
    mutate(p.val = map(spread, p.val.two.group, paired.test = paired.test)) %>%
    select(-data, -spread) %>%
    unnest() %>% ungroup() %>% 
    mutate(iteration = iter) %>%
    mutate(comparison = paste0(filter_events, collapse = " vs "))
  
  return(Data.iter)
}

